/** @file   civilian.h
 * @brief   Declaration of Civial - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_CIVILIAN_H
#define H_WWW_CIVILIAN_H

#include "gameobject.h"


namespace WeWantWar {


/** @class  Civilian
 * @brief   Represents the civians walking in the playfield.
 * @author  Tomi Lamminsaari
 *
 * Civilians are neutral being in the playfield. They just do their own tasks
 * and don't harm @c Player or @c Alien - objects. They can be killed.
 */
class Civilian : public GameObject
{
public:

  /** Declare the SpawnEvent - class to be our friend so that it can access
   * our protected interface.
   */
  friend class SpawnEvent;

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** Possible types of the civians. Only affects on graphics.
   */
  enum CivType {
    /** Civilian 1 : Walking man. (blue) */
    TYPE_CIV1,
    /** Civilian 2 : Smoking man. */
    TYPE_CIV2,
    /** Civilian 3 : Walking woman */
    TYPE_CIV3,
    /** Civilian 4 : Walking man. (cyan) */
    TYPE_CIV4,
    /** Civilian 5 : drinking man */
    TYPE_CIV5,
    /** Civilian 6 : sitting and drinking man */
    TYPE_CIV6,
    /** Civilian 7 : Dancing woman */
    TYPE_DANCER1,
    /** Civilian 8 : Dancing man */
    TYPE_DANCER2
  };
  


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   * @param     ct                Type of this civilian
   */
	Civilian( CivType ct );

  /** Constrctor.
   * @param     ct                Type of this civilian
   * @param     routename         Name of the route we should follow. The
   *                              routes are defined in Map description file.
   */
  Civilian( CivType ct, const std::string& routename );

	/** Destructor
   */
	virtual ~Civilian();

private:

	/** Copy constructor.
   * @param     rO                Reference to another Civilian
   */
  Civilian( const Civilian& rO );

	/** Assignment operator
   * @param     rO                Reference to another Civilian
   * @return    Reference to us.
   */
  Civilian& operator = ( const Civilian& rO );

public:


  ///
  /// Methods
  /// =======

  /** The update method
   */
  virtual void update();
  
  /** Kills this civilian
   */
  virtual void kill();
  
  /** Makes sound
   * @param     id                ID of the sound
   */
  virtual void makeSound( GameObject::SoundID id ) const;
  
  /** Redraws this object
   * @param     pQueue            The redraw queue
   */
  virtual void redraw( RedrawQueue* pQueue );
  
  /** Hit by bullet
   * @param     pB                Pointer to bullet
   */
  virtual bool hitByBullet( Bullet* pB );


  ///
  /// Getter methods
  /// ==============

  /** Tells if this object is reloading.
   * @return    <code>false</code> since civilians don't have weapons.
   */
  virtual bool reloading() const;
  
  /** Returns the type of this object.
   * @return    ObjectID::TYPE_CIVILIAN
   */
  virtual ObjectID::Type objectType() const;


protected:

  /** After you've set civilian type to the @c m_civtype - member, you should
   * call this method. Based on that member, this sets the correct armor,
   * speed and other values.
   */
  void initCivilian();
  
  /** Based on @c m_civtype - member this method sets correct animation
   * for us.
   * @param     animID            The ID of the animation to set. See
   *                              the @c GameAnims - class for more information.
   */
  void setCorrectAnim( int animID );
  
  
  
  ///
  /// Members
  /// =======

  /** The movement vector. */
  eng2d::Vec2D  m_movementVec;
  
  /** Type of this civilian */
  CivType   m_civtype;

private:

  ///
  /// Private members
  /// ===============
  

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: civilian.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:16:51+02  lamminsa
 * Initial revision
 *
 */
